<?php
/**
 * WAAVE Compliance
 *
 * @class       WAAVE_Compliance
 * @package WAAVE_Compliance
 */

/**
 * WAAVE_Compliance
 */
class WAAVE_Compliance {
	const PROD_URL            = 'https://getwaave.co';
	const SANDBOX_URL         = 'https://staging.getwaave.co';
	const API_KEY_GOOGLE_FONT = 'AIzaSyAZnNr3PX2QGG349vh1CXlZ-H9fND82dbo';

	const API_KEY_DESCRIPTION = 'WAAVE API Key';

	/**
	 * Instance.
	 *
	 * @var this
	 */
	public static $instance;

	/**
	 * Function init.
	 */
	public static function init() {
		if ( ! self::$instance ) {
			self::$instance = new WAAVE_Compliance();
		}

		return self::$instance;
	}

	/**
	 * Function plugin_deactivation.
	 */
	public static function plugin_deactivation() {
		global $wpdb;

		// phpcs:ignore
		$wpdb->delete(
			$wpdb->prefix . 'woocommerce_api_keys',
			array( 'description' => self::API_KEY_DESCRIPTION ),
			array( '%s' )
		);
	}

	/**
	 * Function construct.
	 */
	public function __construct() {
		add_filter( 'plugin_action_links_' . plugin_basename( WAAVE_COMPLIANCE_MAIN_FILE ), array( $this, 'add_plugin_links' ) );
		add_filter( 'woocommerce_settings_tabs_array', array( $this, 'add_settings_tab' ), 99 );
		add_action( 'woocommerce_settings_waave', array( $this, 'settings_tab' ) );
		add_action( 'woocommerce_update_options_waave', array( $this, 'update_settings' ) );
		add_action(
			'woocommerce_update_options',
			function() {
				add_action(
					'admin_notices',
					function() {
						echo '<div class="notice notice-warning is-dismissible">
						<p><strong>Warning:</strong> Please clear the cache of system and reload the website!</p>
                          </div>';
					}
				);
			}
		);

		// Card #894, plugin activation.
		add_action( 'admin_footer', array( $this, 'add_settings_popup' ) );
		add_action( 'wp_ajax_waave_compliance_save_settings_popup', array( $this, 'save_settings_popup' ) );

		$plugin_data = get_plugin_data( WAAVE_COMPLIANCE_MAIN_FILE );
		$version     = $plugin_data['Version'];
		wp_enqueue_script( 'age-gate-configuration', plugins_url( '../assets/js/age-gate-configuration.js', __FILE__ ), array( 'jquery' ), $version, true );
		wp_enqueue_style( 'age-gate-configuration', plugins_url( '../assets/css/age-gate-configuration.css', __FILE__ ), array(), $version );
		wp_localize_script(
			'age-gate-configuration',
			'age_gate_config',
			array(
				'ageGatePopupDesignMode'         => get_option( 'waave_compliance_age_gate_popup_design_mode' ),
				'ageGatePopupSize'               => get_option( 'waave_compliance_age_gate_popup_size' ),
				'ageGatePopupButtonColorDefault' => get_option( 'waave_compliance_age_gate_popup_button_color_default' ),
				'ageGatePopupLogoDefault'        => get_option( 'waave_compliance_age_gate_popup_logo_default' ),
				'ageGatePopupFontFamilyType'     => get_option( 'waave_compliance_age_gate_popup_font_family_type' ),
				'ageGatePopupMode'               => get_option( 'waave_compliance_age_gate_mode' ),
			)
		);
	}

	/**
	 * Adds plugin page links
	 *
	 * @param array $links all plugin links.
	 * @return array $links all plugin links + our custom links (i.e., "Settings")
	 */
	public function add_plugin_links( $links ) {
		$plugin_links = array(
			'<a href="' . admin_url( 'admin.php?page=wc-settings&tab=waave' ) . '">Settings</a>',
		);

		return array_merge( $plugin_links, $links );
	}

	/**
	 * Function add_settings_tab.
	 *
	 * @param array $settings_tabs settings_tabs.
	 */
	public function add_settings_tab( $settings_tabs ) {
		$settings_tabs['waave'] = 'WAAVE';
		return $settings_tabs;
	}

	/**
	 * Function settings_tabs.
	 */
	public function settings_tab() {
		woocommerce_admin_fields( $this->get_settings() );
	}

	/**
	 * Function update_settings.
	 */
	public function update_settings() {
		woocommerce_update_options( $this->get_settings() );
		$plugin_data = get_plugin_data( WAAVE_COMPLIANCE_MAIN_FILE );
		$version     = $plugin_data['Version'];
		wp_enqueue_script( 'age-gate-configuration', plugins_url( '../assets/js/age-gate-configuration.js', __FILE__ ), array( 'jquery' ), $version, true );
		wp_localize_script(
			'age-gate-configuration',
			'age_gate_config',
			array(
				'ageGatePopupDesignMode'         => get_option( 'waave_compliance_age_gate_popup_design_mode' ),
				'ageGatePopupSize'               => get_option( 'waave_compliance_age_gate_popup_size' ),
				'ageGatePopupButtonColorDefault' => get_option( 'waave_compliance_age_gate_popup_button_color_default' ),
				'ageGatePopupLogoDefault'        => get_option( 'waave_compliance_age_gate_popup_logo_default' ),
				'ageGatePopupFontFamilyType'     => get_option( 'waave_compliance_age_gate_popup_font_family_type' ),
				'ageGatePopupMode'               => get_option( 'waave_compliance_age_gate_mode' ),
			)
		);
		$this->call_waave_active_api();
		$time_remerber_me = 10 * 365 * 24 * 60 * 60; // 10 years
		setcookie( 'age_verification_mode', get_option( 'waave_compliance_age_gate_mode' ) === 'yes' ? 1 : 0, time() + $time_remerber_me, COOKIEPATH, COOKIE_DOMAIN );
	}

	/**
	 * Function add_settings_popup
	 *
	 * @return void
	 */
	public function add_settings_popup() {
		if ( ! get_option( 'waave_compliance_show_setup_popup' ) ) {
			return;
		}
		?>
		<div id="waave-compliance-setup-popup">
			<div class="popup-overlay"></div>
			<div class="popup-content">
				<h2>WAAVE Compliance</h2>
				<form id="waave-compliance-setup-form" method="post">
				<?php wp_nonce_field( 'waave_compliance_setup_form_key', 'my_plugin_nonce' ); ?>

					<div id="waave-form-message" style="display: none;"></div>
					<table class="form-table">
						<tr>
							<th scope="row">
								<label for="client_id">Email <span class="required">*</span></label>
							</th>
							<td>
								<input type="text" name="client_id" id="client_id" class="regular-text" required>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="client_password">Password <span class="required">*</span></label>
							</th>
							<td class="password-input">
								<input type="password" name="client_password" id="client_password" class="regular-text" required>
								<span class="show-password-input"></span>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="testmode">Sandbox</label>
							</th>
							<td>
								<input type="checkbox" name="testmode" id="testmode">
							</td>
						</tr>
					</table>
					<div id="waave-form-buttons" class="submit">
						<button type="submit" class="button button-primary">Save</button>
						<button type="button" id="waave-close-button" class="button">Close</button>
						<span id="waave-loading-spinner" class="spinner" style="visibility: hidden;"></span>
					</div>
				</form>
			</div>
		</div>
		<?php
		delete_option( 'waave_compliance_show_setup_popup' );
	}

	/**
	 * Function save_settings_popup
	 *
	 * @return void
	 */
	public function save_settings_popup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ) );
		}

		$is_valid = check_ajax_referer( 'waave_compliance_setup_form_key', 'nonce' );

		if ( ! $is_valid ) {
			wp_send_json_error( array( 'message' => 'The nonce is invalid' ) );
		}

		if ( ! isset( $_POST['client_id'] ) || ! isset( $_POST['client_password'] ) ) {
			wp_send_json_error( array( 'message' => 'Invalid credentials!' ) );
		}

		$client_id       = sanitize_text_field( wp_unslash( $_POST['client_id'] ) );
		$client_password = sanitize_text_field( wp_unslash( $_POST['client_password'] ) );
		$testmode        = isset( $_POST['testmode'] ) && '1' === $_POST['testmode'] ? 'yes' : 'no';

		$response = $this->call_waave_verify_api( $client_id, $client_password, $testmode );

		if ( isset( $response['message'] ) ) {
			wp_send_json_error( $response );
		}

		if ( empty( $response['venue_id'] ) ) {
			wp_send_json_error( array( 'message' => 'Invalid credentials!' ) );
		}

		update_option( 'waave_compliance_venue_id', $response['venue_id'] );
		update_option( 'waave_compliance_client_password', $response['venue_password'] );
		update_option( 'waave_compliance_testmode', $testmode );

		$response = $this->call_waave_active_api();
		if ( $response ) {
			wp_send_json_error( array( 'message' => 'Invalid credentials!' ) );
		}

		$result = $this->generate_api_key();
		if ( isset( $result['message'] ) ) {
			wp_send_json_error( $result );
		}

		$consumer_key    = $result['consumer_key'];
		$consumer_secret = $result['consumer_secret'];

		update_option( 'waave_compliance_consumer_key', $consumer_key );
		update_option( 'waave_compliance_consumer_secret', $consumer_secret );

		update_option( 'waave_compliance_age_gate_mode', 'yes' );
		update_option( 'waave_compliance_disclaimers_mode', 'yes' );
		update_option( 'waave_compliance_footer_mode', 'yes' );
		update_option( 'waave_compliance_disclaimers_background_color', '#fff' );
		update_option( 'waave_compliance_disclaimers_color', '#111' );
		update_option( 'waave_compliance_age_gate_popup_design_mode', 'no' );
		update_option( 'waave_compliance_age_gate_popup_logo_default', 'yes' );
		update_option( 'waave_compliance_age_gate_popup_logo_width', 350 );
		update_option( 'waave_compliance_age_gate_popup_logo_height', 350 );
		update_option( 'waave_compliance_age_gate_popup_background_color', '#fff' );
		update_option( 'waave_compliance_age_gate_popup_color', '#000' );
		update_option( 'waave_compliance_age_gate_popup_button_color_default', 'yes' );
		update_option( 'waave_compliance_age_gate_popup_button_color', '#000' );

		wp_send_json_success( array( 'message' => 'WAAVE Compliance settings saved successfully!' ) );
	}

	/**
	 * Function generate api keys
	 *
	 * @return array
	 */
	private function generate_api_key() {
		$consumer_key    = 'ck_' . wc_rand_hash();
		$consumer_secret = 'cs_' . wc_rand_hash();

		global $wpdb;

		$data = array(
			'user_id'         => get_current_user_id(),
			'description'     => self::API_KEY_DESCRIPTION,
			'permissions'     => 'read',
			'consumer_key'    => wc_api_hash( $consumer_key ),
			'consumer_secret' => $consumer_secret,
			'truncated_key'   => substr( $consumer_key, -7 ),
		);

		// phpcs:ignore
		$wpdb->insert(
			$wpdb->prefix . 'woocommerce_api_keys',
			$data,
			array(
				'%d',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
			)
		);

		if ( 0 === $wpdb->insert_id ) {
			return array(
				'message' => 'There was an error generating your API Key.',
			);
		}

		$data['consumer_key'] = $consumer_key;

		return $data;
	}

	/**
	 * Function get_settings.
	 */
	private function get_settings() {
		$google_font_api_url = 'https://www.googleapis.com/webfonts/v1/webfonts?key=' . self::API_KEY_GOOGLE_FONT;
		$request             = wp_remote_get( $google_font_api_url );
		$response            = json_decode( wp_remote_retrieve_body( $request ), true );
		$google_fonts        = array();
		foreach ( $response['items'] as $font ) {
			$google_fonts[ $font['family'] ] = $font['family'];
		}

		return array(
			'section_title'                       => array(
				'name' => 'REST API',
				'type' => 'title',
				'desc' => '',
				'id'   => 'waave_compliance_section_title',
			),
			'consumer_key'                        => array(
				'name'              => 'Consumer key',
				'type'              => 'text',
				'desc'              => '',
				'custom_attributes' => array(
					'readonly' => true,
				),
				'id'                => 'waave_compliance_consumer_key',
			),
			'consumer_secret'                     => array(
				'name'              => 'Consumer secret',
				'type'              => 'password',
				'desc'              => '',
				'custom_attributes' => array(
					'readonly' => true,
				),
				'id'                => 'waave_compliance_consumer_secret',
			),
			'section_end'                         => array(
				'type' => 'sectionend',
				'id'   => 'waave_compliance_section_end',
			),
			'section_client_title'                => array(
				'name' => 'Client',
				'type' => 'title',
				'desc' => '',
				'id'   => 'waave_compliance_section_client_title',
			),
			'venue_id'                            => array(
				'name'              => 'Venue ID',
				'type'              => 'text',
				'desc'              => '',
				'custom_attributes' => array(
					'readonly' => true,
				),
				'id'                => 'waave_compliance_venue_id',
			),
			'client_password'                     => array(
				'name'              => 'Password',
				'type'              => 'password',
				'desc'              => '',
				'custom_attributes' => array(
					'readonly' => true,
				),
				'id'                => 'waave_compliance_client_password',
			),
			'testmode'                            => array(
				'name'    => 'WAAVE Sandbox',
				'type'    => 'checkbox',
				'id'      => 'waave_compliance_testmode',
				'default' => 'yes',
			),
			'section_client_end'                  => array(
				'type' => 'sectionend',
				'id'   => 'waave_compliance_section_client_end',
			),
			'section_age_gate_title'              => array(
				'name' => 'Age Gate & Disclaimers',
				'type' => 'title',
				'desc' => '',
				'id'   => 'waave_compliance_age_gate_title',
			),
			'age_gate_mode'                       => array(
				'name'    => 'WAAVE Age Gate Popup Mode',
				'type'    => 'checkbox',
				'id'      => 'waave_compliance_age_gate_mode',
				'desc'    => 'Enable/Disable',
				'default' => 'yes',
			),
			'disclaimers_mode'                    => array(
				'name'    => 'WAAVE Disclaimers Mode',
				'type'    => 'checkbox',
				'id'      => 'waave_compliance_disclaimers_mode',
				'desc'    => 'Enable/Disable',
				'default' => 'yes',
			),
			'footer_mode'                         => array(
				'name'    => 'WAAVE Footer Mode',
				'type'    => 'checkbox',
				'id'      => 'waave_compliance_footer_mode',
				'desc'    => 'Enable/Disable',
				'default' => 'yes',
			),
			'disclaimers_background_color'        => array(
				'name'    => '',
				'type'    => 'color',
				'id'      => 'waave_compliance_disclaimers_background_color',
				'desc'    => 'Background Waave Footer Color',
				'default' => '#fff',
				'class'   => 'inputcolor ',
			),
			'disclaimers_color'                   => array(
				'name'    => '',
				'type'    => 'color',
				'id'      => 'waave_compliance_disclaimers_color',
				'desc'    => 'Waave Footer Color',
				'default' => '#111',
				'class'   => 'inputcolor ',
			),
			'age_gate_popup_design_mode'          => array(
				'name'    => 'WAAVE Age Gate Popup Design Mode',
				'type'    => 'checkbox',
				'id'      => 'waave_compliance_age_gate_popup_design_mode',
				'desc'    => 'Enable/Disable',
				'default' => 'no',
			),
			'age_gate_popup_logo_default'         => array(
				'name'    => '',
				'type'    => 'checkbox',
				'id'      => 'waave_compliance_age_gate_popup_logo_default',
				'desc'    => 'Default Logo',
				'default' => 'yes',
			),
			'age_gate_popup_logo'                 => array(
				'name'     => 'WAAVE Age Gate Popup Logo',
				'type'     => 'url',
				'desc'     => '',
				'id'       => 'waave_compliance_age_gate_popup_logo',
				'desc_tip' => 'Paste the URL of an image you want to show in the age gate popup. Upload images using the media uploader (Media > Add New).',
			),
			'age_gate_popup_logo_width'           => array(
				'name'    => 'WAAVE Age Gate Popup Logo Width (px)',
				'type'    => 'number',
				'desc'    => '',
				'id'      => 'waave_compliance_age_gate_popup_logo_width',
				'default' => 350,
			),
			'age_gate_popup_logo_height'          => array(
				'name'    => 'WAAVE Age Gate Popup Logo Height (px)',
				'type'    => 'number',
				'desc'    => '',
				'id'      => 'waave_compliance_age_gate_popup_logo_height',
				'default' => 350,
			),
			'age_gate_popup_size'                 => array(
				'name'    => 'WAAVE Age Gate Popup Size',
				'type'    => 'select',
				'id'      => 'waave_compliance_age_gate_popup_size',
				'desc'    => '',
				'default' => '',
				'options' => array(
					'large'     => 'Large',
					'medium'    => 'Medium',
					'small'     => 'Small',
					'customize' => 'Customize',
				),
			),
			'age_gate_popup_width'                => array(
				'name' => 'WAAVE Age Gate Popup Width (px)',
				'type' => 'number',
				'desc' => '',
				'id'   => 'waave_compliance_age_gate_popup_width',
			),
			'age_gate_popup_height'               => array(
				'name' => 'WAAVE Age Gate Popup Height (px)',
				'type' => 'number',
				'desc' => '',
				'id'   => 'waave_compliance_age_gate_popup_height',
			),
			'age_gate_popup_background_color'     => array(
				'name'    => '',
				'type'    => 'color',
				'id'      => 'waave_compliance_age_gate_popup_background_color',
				'desc'    => 'WAAVE Age Gate Popup Background Color',
				'default' => '#ffffff',
				'class'   => 'inputcolor ',
			),
			'age_gate_popup_color'                => array(
				'name'    => '',
				'type'    => 'color',
				'id'      => 'waave_compliance_age_gate_popup_color',
				'desc'    => 'WAAVE Age Gate Popup Font Color',
				'default' => '#000000',
				'class'   => 'inputcolor ',
			),
			'age_gate_popup_font_size'            => array(
				'name'    => 'WAAVE Age Gate Popup Font Size',
				'type'    => 'select',
				'id'      => 'waave_compliance_age_gate_popup_font_size',
				'desc'    => '',
				'default' => '',
				'options' => array(
					'8'  => 8,
					'9'  => 9,
					'10' => 10,
					'11' => 11,
					'12' => 12,
					'14' => 14,
					'16' => 16,
					'18' => 18,
					'20' => 20,
					'22' => 22,
					'24' => 24,
					'26' => 26,
					'28' => 28,
					'36' => 36,
					'48' => 48,
					'72' => 72,
				),
			),
			'age_gate_popup_font_family_type'     => array(
				'name'    => 'WAAVE Age Gate Popup Font Family Type',
				'type'    => 'select',
				'id'      => 'waave_compliance_age_gate_popup_font_family_type',
				'desc'    => '',
				'default' => '',
				'options' => array(
					'default'     => 'Default',
					'google_font' => 'Google Font',
				),
			),
			'age_gate_popup_google_font'          => array(
				'name'    => 'WAAVE Age Gate Popup Font Family',
				'type'    => 'select',
				'id'      => 'waave_compliance_age_gate_popup_google_font',
				'desc'    => '',
				'default' => '',
				'options' => $google_fonts,
			),
			'age_gate_popup_font_family'          => array(
				'name' => 'WAAVE Age Gate Popup Font Family',
				'type' => 'text',
				'desc' => '',
				'id'   => 'waave_compliance_age_gate_popup_font_family',
			),
			'age_gate_popup_button_color_default' => array(
				'name'    => '',
				'type'    => 'checkbox',
				'id'      => 'waave_compliance_age_gate_popup_button_color_default',
				'desc'    => 'Default Button Color',
				'default' => 'yes',
			),
			'age_gate_popup_button_color'         => array(
				'name'    => '',
				'type'    => 'color',
				'id'      => 'waave_compliance_age_gate_popup_button_color',
				'desc'    => 'WAAVE Age Gate Popup Button Color',
				'default' => '#000000',
				'class'   => 'inputcolor ',
			),
			'section_age_gate_end'                => array(
				'type' => 'sectionend',
				'id'   => 'waave_compliance_age_gate_section_end',
			),
		);
	}

	/**
	 * Function call_waave_active_api.
	 */
	private function call_waave_active_api() {
		$plugin_data = get_plugin_data( WAAVE_COMPLIANCE_MAIN_FILE );
		$body        = array(
			'version'              => $plugin_data['Version'],
			'ping_url'             => get_rest_url( null, 'waave-compliance/v1/version' ),
			'category_url'         => get_rest_url( null, 'waave-compliance/v1/products/categories' ),
			'menu_item_url'        => get_rest_url( null, 'waave-compliance/v1/products' ),
			'disclaimer_sync_url'  => get_rest_url( null, 'waave-compliance/v1/disclaimer/sync' ),
			'menu_item_detail_url' => get_rest_url( null, 'waave-compliance/v1/product' ),
			'venue_id'             => get_option( 'waave_compliance_venue_id' ),
			'password'             => get_option( 'waave_compliance_client_password' ),
		);

		$url = self::PROD_URL;
		if ( 'yes' === get_option( 'waave_compliance_testmode' ) ) {
			$url = self::SANDBOX_URL;
		}

		$url .= '/compliance/active';

		$options = array(
			'body' => $body,
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		if ( empty( $response['success'] ) && $response['message'] ) {
			if ( wp_doing_ajax() ) {
				return $response;
			}

			WC_Admin_Settings::add_error( $response['message'] );
		}
	}

	/**
	 * Function call_waave_verify_api
	 *
	 * @param string $client_id client id.
	 * @param string $client_password client password.
	 * @param string $testmode test mode.
	 */
	private function call_waave_verify_api( $client_id, $client_password, $testmode ) {
		$url = self::PROD_URL;
		if ( 'yes' === $testmode ) {
			$url = self::SANDBOX_URL;
		}

		$url .= '/compliance/verify';

		$options = array(
			'body' => array(
				'email'    => $client_id,
				'password' => $client_password,
			),
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		return $response;
	}
}
