<?php
/**
 * WAAVE Checkout
 *
 * @class       WC_WAAVE_Checkout
 * @extends     WC_WAAVE_Complete
 * @package     WAAVE Complete
 */

/**
 * WC_WAAVE_Checkout
 */
class WC_WAAVE_Checkout extends WC_WAAVE_Complete {

	const PROD_VGS_ENV         = 'live';
	const PROD_VGS_VAULT_ID    = 'tntf3oywshf';
	const SANDBOX_VGS_ENV      = 'sandbox';
	const SANDBOX_VGS_VAULT_ID = 'tntflxlh3fw';

	const AGE_CHECKER_URL         = 'https://api.agechecker.net/';
	const SANDBOX_AGE_CHECKER_KEY = 'Hix8qri6gxpJOP3PSHsmOilzehv8OTpI';
	const PROD_AGE_CHECKER_KEY    = 'R315tlFOdf03jMXBCgutFCkLcCBTShcH';

	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {
		parent::__construct();

		$this->id                 = 'waave_checkout';
		$this->method_title       = 'WAAVE Complete';
		$this->method_description = 'WAAVE Checkout';

		// Load the settings.
		$this->init_form_fields();
		$this->init_settings();

		// For AgeCheck.
		$this->vgs_env         = self::PROD_VGS_ENV;
		$this->vgs_vault_id    = self::PROD_VGS_VAULT_ID;
		$this->age_checker_key = self::PROD_AGE_CHECKER_KEY;
		if ( 'yes' === $this->get_option( 'testmode' ) ) {
			$this->vgs_env         = self::SANDBOX_VGS_ENV;
			$this->vgs_vault_id    = self::SANDBOX_VGS_VAULT_ID;
			$this->age_checker_key = self::SANDBOX_AGE_CHECKER_KEY;
		}

		// Common fields.
		$this->title       = 'WAAVECheckout: Register / Login to pay with your WAAVE Account';
		$this->description = '<p>Payment protection program on all your transactions. US & International major credit and debit cards. More about <a href="https://www.getwaave.com/what-is-waave" target="blank">WAAVE</a></p>' .
		'<p style="margin-top: 10px">' . $this->get_option( 'description' ) . '</p>';

		// Enable/Disable WAAVE Checkout.
		$this->enabled = $this->get_option( 'enabled', 'yes' );

		// Actions.
		add_action( 'woocommerce_update_options_payment_gateways_waave_checkout', array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_checkout_create_order', array( $this, 'before_checkout_create_order' ) );
		add_action( 'woocommerce_receipt_waave_checkout', array( $this, 'receipt_page' ) );
		add_action( 'woocommerce_api_wc_waave_checkout', array( $this, 'check_waave_response' ) );
		add_action( 'woocommerce_update_options_checkout_waave_checkout', array( $this, 'update_waave_checkout' ) );
		add_action( 'woocommerce_thankyou_waave_checkout', array( $this, 'thankyou_page' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'plugin_vgs_assets' ) );
		add_filter( 'woocommerce_get_checkout_payment_url', array( $this, 'filter_payment_url' ) );
	}

	/**
	 * {@inheritdoc}
	 */
	public function init_form_fields() {
		$this->form_fields = apply_filters(
			'wc_waave_form_fields',
			array(

				'enabled'     => array(
					'title'   => 'Enable/Disable',
					'type'    => 'checkbox',
					'label'   => 'Enable WAAVE Checkout',
					'default' => 'yes',
				),

				'testmode'    => array(
					'title'       => 'WAAVE Sandbox',
					'type'        => 'checkbox',
					'label'       => 'WAAVE Sandbox',
					'description' => 'Place the payment gateway in the development mode.',
					'default'     => 'yes',
				),

				'access_key'  => array(
					'title'             => 'Access Key',
					'type'              => 'text',
					'description'       => 'This is an access key for WAAVE connection.',
					'desc_tip'          => true,
					'custom_attributes' => array(
						'readonly' => true,
					),
				),

				'private_key' => array(
					'title'             => 'Private Key',
					'type'              => 'text',
					'description'       => 'This is an private key for WAAVE connection.',
					'desc_tip'          => true,
					'custom_attributes' => array(
						'readonly' => true,
					),
				),

				'venue_id'    => array(
					'title'             => 'Venue ID',
					'type'              => 'text',
					'description'       => 'This is a venue id for WAAVE connection.',
					'desc_tip'          => true,
					'custom_attributes' => array(
						'readonly' => true,
					),
				),

				'description' => array(
					'title'       => 'Description',
					'type'        => 'textarea',
					'description' => 'This will appear below WAAVE payemt button.',
					'desc_tip'    => true,
				),
			)
		);
	}

	/**
	 * Function filter_payment_url.
	 *
	 * @param string $pay_url pay url.
	 */
	public function filter_payment_url( $pay_url ) {
		return add_query_arg( 'validation_code', $this->validation_code, $pay_url );
	}

	/**
	 * Function plugin_vgs_assets.
	 */
	public function plugin_vgs_assets() {
		$plugin_data = get_plugin_data( WC_WAAVE_MAIN_FILE );
		$version     = $plugin_data['Version'];
		wp_enqueue_script( 'woocommerce_waave_vgs_plugin', 'https://js.verygoodvault.com/vgs-collect/2.18.1/vgs-collect.js', array(), $version, false );
		wp_enqueue_script( 'woocommerce_waave_vgs_add_card', plugins_url( 'assets/js/vgs_add_card.js', WC_WAAVE_MAIN_FILE ), array(), $version, true );

		$products = $this->get_cart_products();
		$vault_id = base64_encode( $this->vgs_vault_id ); // phpcs:ignore

		wp_localize_script(
			'woocommerce_waave_vgs_plugin',
			'vgs_config',
			array(
				'env'              => $this->vgs_env,
				'vault_id'         => $vault_id,
				'waave_form_id'    => 'waave_guest_credit_card',
				'products'         => wp_json_encode( $products ),
				'waave_url'        => $this->api_url,
				'age_checker_mode' => 1,
				'age_checker'      => $this->age_checker_key,
				'waave_logo'       => esc_url( WAAVE_COMPLETE_PLUGIN_URL . '/assets/images/logo.png' ),
			)
		);
	}

	/**
	 * Before checkout create order.
	 *
	 * @param WC_Order $order Order object.
	 * @throws Exception When error message responsed.
	 */
	public function before_checkout_create_order( $order ) {
		if ( $this->id !== $order->get_payment_method() ) {
			return;
		}

		if ( ! $this->age_check_id ) {
			return;
		}

		$body = array(
			'age_check_id'    => $this->age_check_id,
			'validation_code' => $this->validation_code,
		);

		$url     = $this->api_url . '/compliance/age-check-status';
		$options = array(
			'body' => $body,
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		if ( isset( $response['message'] ) && empty( $response['success'] ) ) {
			throw new Exception( $response['message'] );
		}
	}

	/**
	 * {@inheritdoc}
	 *
	 * @param string $order_id order_id.
	 * @throws Exception When error message responsed.
	 */
	public function process_payment( $order_id ) {
		$order = wc_get_order( $order_id );

		$billing  = $this->get_order_billing( $order );
		$shipping = $this->get_order_shipping( $order );

		$body = array(
			'venue_id'   => $this->venue_id,
			'amount'     => $order->get_total(),
			'email'      => $billing['billing_email'],
			'ip_address' => $order->get_customer_ip_address(),
			'billing'    => $billing,
			'shipping'   => $shipping,
		);

		$url     = $this->api_url . '/compliance/validate-waavesonar';
		$options = array(
			'body' => $body,
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		if ( isset( $response['message'] ) && empty( $response['success'] ) ) {
			throw new Exception( $response['message'] );
		}

		// Return to receipt page.
		return array(
			'result'   => 'success',
			'redirect' => $order->get_checkout_payment_url( true ),
		);
	}

	/**
	 * Function update_waave_checkout.
	 */
	public function update_waave_checkout() {
		$api_url = rest_url( 'waave-complete/v1/waavecheckout-state/change' );
		$options = array(
			'body' => array(
				'api_url' => $api_url,
			),
		);

		$url = $this->api_url . '/complete/waavecheckout/' . $this->venue_id;
		wp_remote_post( $url, $options );
	}

	/**
	 * Reciept page.
	 * Display text and a button to direct the user to WAAVE.
	 *
	 * @since 1.0.0
	 * @param string $order_id order id.
	 */
	public function receipt_page( $order_id ) {
		?>
		<p>Thank you for your order, please click the button below to pay with WAAVE.</p>

		<?php

		$order           = wc_get_order( $order_id );
		$validation_code = isset( $_GET['validation_code'] ) ? wp_unslash( $_GET['validation_code'] ) : ''; // phpcs:ignore

		$data_to_send = array(
			// WAAVE details.
			'access_key'      => $this->access_key,
			'return_url'      => $this->get_return_url( $order ),
			'cancel_url'      => wc_get_cart_url(),
			'callback_url'    => $this->callback_url,

			// Order details.
			'amount'          => $order->get_total(),
			'reference_id'    => $order->get_id(),
			'currency'        => $order->get_currency(),
			'shipping_cost'   => $order->get_shipping_total(),

			// Inconsitent data.
			'venue_id'        => $this->venue_id,
			'validation_code' => $validation_code,
		);
		?>

		<form action="<?php echo esc_url( $this->url . '/waavepay/checkout' ); ?>" id="waave_payment_form">
			<?php
			foreach ( $data_to_send as $key => $value ) {
				?>
				<input type="hidden" name="<?php echo esc_attr( $key ); ?>" value="<?php echo esc_attr( $value ); ?>" />
				<?php
			}
			?>
			<input type="submit" class="button-alt" id="submit_waave_payment_form" value="Pay via WAAVE" />
			<a class="button cancel" href="<?php echo esc_url( $order->get_cancel_order_url() ); ?>">Cancel order &amp; restore cart</a>
			<script type="text/javascript">
				jQuery(function(){
					jQuery("body").block({
						message: "Thank you for your order. We are now redirecting you to WAAVE to make payment.",
						overlayCSS: {
							background: "#fff",
							opacity: 0.6
						},
						css: {
							padding: 20,
							color: "#555",
							textAlign: "center",
							border: "3px solid #aaa",
							backgroundColor: "#fff",
							cursor: "wait",
						}
					});

					jQuery( "#submit_waave_payment_form" ).click();
				});
			</script>
		</form>
		<?php
	}

	/**
	 * Check WAAVE response.
	 */
	public function check_waave_response() {
		$json = file_get_contents( 'php://input' );
		$data = stripslashes_deep( json_decode( $json, true ) );

		ob_end_clean();

		$valid = $this->validate_signature( $data );
		if ( ! $valid ) {
			wp_send_json(
				array(
					'success' => false,
					'message' => 'Signature is invalid',
				)
			);
		}

		$order_id = absint( $data['reference_id'] );
		$order    = wc_get_order( $order_id );

		if ( ! $order ) {
			wp_send_json(
				array(
					'success' => false,
					'message' => 'Order is invalid',
				)
			);
		}

		// phpcs:ignore
		if ( $data['amount'] != $order->get_total() || $this->is_successful_order( $order ) ) {
			wp_send_json(
				array(
					'success' => false,
					'message' => 'Order amount is incorrect or status is completed',
				)
			);
		}

		switch ( $data['status'] ) {
			case 'completed':
				$order->add_order_note( 'WAAVE payment completed' );
				$order->payment_complete();
				break;
			case 'pending':
				$order->update_status( 'on-hold', 'This payment is pending via WAAVE.' );
				break;
			case 'cancelled':
				$order->update_status( 'cancelled', 'This payment has cancelled via WAAVE.' );
				break;
			case 'failed':
				$order->update_status( 'failed', 'This payment is failed via WAAVE.' );
				break;
			default:
				break;
		}

		$this->log( 'Successfully updated status for order: ' . $order_id );
		$this->log( 'Request data: ' . $json );
		wp_send_json( array( 'success' => true ) );
	}

	/**
	 * Function change_waave_checkout_state.
	 *
	 * @param Request $request request.
	 */
	public static function change_waave_checkout_state( $request ) {
		$enabled = $request->get_param( 'enabled' );

		$settings            = get_option( 'woocommerce_waave_checkout_settings' );
		$settings['enabled'] = $enabled;
		update_option( 'woocommerce_waave_checkout_settings', $settings );

		wp_send_json( array( 'success' => true ) );
	}

	/**
	 * Validate signature.
	 *
	 * @param array $data data.
	 */
	private function validate_signature( $data ) {
		$url  = $this->callback_url;
		$body = wp_json_encode( $data );

		$signature        = hash( 'sha256', $this->private_key . $url . $body );
		$header_signature = isset( $_SERVER['HTTP_X_API_SIGNATURE'] ) ? wp_unslash( $_SERVER['HTTP_X_API_SIGNATURE'] ) : ''; // phpcs:ignore

		if ( $signature === $header_signature ) {
			return true;
		}

		$this->log( 'Signature is invalid.' );
		$this->log( 'Callback url: ' . $url );
		$this->log( 'Request body: ' . $body );

		return false;
	}

	/**
	 * Get cart products.
	 */
	private static function get_cart_products() {
		$products = array();

		$cart_contents = WC()->cart->get_cart();
		foreach ( $cart_contents as $value ) {
			$product = $value['data'];
			if ( empty( $product ) ) {
				continue;
			}

			$product_id = $product->get_id();
			$categories = $product->get_category_ids();
			if ( 'variation' === $product->get_type() ) {
				$product_id = $product->get_parent_id();
				$parent     = wc_get_product( $product_id );
				$categories = $parent->get_category_ids();
			}

			$products[] = array(
				'id'         => $product_id,
				'name'       => $product->get_name(),
				'sku'        => $product->get_sku(),
				'price'      => $product->get_price(),
				'quantity'   => $value['quantity'],
				'categories' => $categories,
			);

			return $products;
		}
	}

	/**
	 * Get billing address
	 *
	 * @param Order $order Order.
	 */
	private function get_order_billing( $order ) {
		return array(
			'billing_first_name' => $order->get_billing_first_name(),
			'billing_last_name'  => $order->get_billing_last_name(),
			'billing_email'      => $order->get_billing_email(),
			'billing_country'    => $order->get_billing_country(),
			'billing_address_1'  => $order->get_billing_address_1(),
			'billing_address_2'  => $order->get_billing_address_2(),
			'billing_city'       => $order->get_billing_city(),
			'billing_state'      => ! empty( $order->get_billing_state() ) ? $order->get_billing_state() : 'Municipality',
			'billing_postcode'   => $order->get_billing_postcode(),
			'billing_phone'      => $order->get_billing_phone(),
		);
	}

	/**
	 * Get shipping address
	 *
	 * @param Order $order Order.
	 */
	private function get_order_shipping( $order ) {
		return array(
			'shipping_first_name' => $order->get_shipping_first_name(),
			'shipping_last_name'  => $order->get_shipping_last_name(),
			'shipping_country'    => $order->get_shipping_country(),
			'shipping_address_1'  => $order->get_shipping_address_1(),
			'shipping_address_2'  => $order->get_shipping_address_2(),
			'shipping_city'       => $order->get_shipping_city(),
			'shipping_state'      => ! empty( $order->get_shipping_state() ) ? $order->get_shipping_state() : 'Municipality',
			'shipping_postcode'   => $order->get_shipping_postcode(),
		);
	}
}
