<?php
/**
 * WAAVE Complete
 *
 * @class       WC_WAAVE_Complete
 * @extends     WC_Payment_Gateway
 * @package     WAAVE Complete
 */

/**
 * WC_WAAVE_Credit_Card_Direct
 */
abstract class WC_WAAVE_Complete extends WC_Payment_Gateway {

	const PROD_URL    = 'https://pg.getwaave.co';
	const SANDBOX_URL = 'https://staging-pg.getwaave.co';

	const API_PROD_URL    = 'https://getwaave.co';
	const API_SANDBOX_URL = 'https://staging.getwaave.co';

	const API_TRANS_PROD_URL    = 'https://paymentapi.getwaave.co';
	const API_TRANS_SANDBOX_URL = 'https://paymentapi-staging.getwaave.co';

	/**
	 * $logger
	 *
	 * @var Logger
	 */
	protected $logger;

	/**
	 * $validation_code
	 *
	 * @var string
	 */
	protected $validation_code;

	/**
	 * $age_check_id
	 *
	 * @var string
	 */
	protected $age_check_id;

	/**
	 * $order_button_text
	 *
	 * @var string
	 */
	public $order_button_text = 'Place order';

	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {
		$this->url           = self::PROD_URL;
		$this->api_url       = self::API_PROD_URL;
		$this->api_trans_url = self::API_TRANS_PROD_URL;

		$settings = get_option( 'woocommerce_waave_checkout_settings' );
		if ( 'yes' === $settings['testmode'] ) {
			$this->url           = self::SANDBOX_URL;
			$this->api_url       = self::API_SANDBOX_URL;
			$this->api_trans_url = self::API_TRANS_SANDBOX_URL;
		}

		$this->access_key  = $settings['access_key'];
		$this->private_key = $settings['private_key'];

		$this->venue_id = get_option( 'waave_compliance_venue_id' );
		if ( is_plugin_inactive( 'waave-compliance/waave-compliance.php' ) ) {
			$this->venue_id = $settings['venue_id'];
		}

		$this->callback_url = add_query_arg( 'wc-api', 'WC_WAAVE_Checkout', home_url( '/' ) );
	}

	/**
	 * {@inheritdoc}
	 */
	public function get_icon() {
		$logo_url  = 'https://getwaave.co/images/plugin_logo.png';
		$icons_str = '<img src="' . $logo_url . '" class="" width="100" alt="WAAVE" />';
		return apply_filters( 'woocommerce_gateway_icon', $icons_str, $this->id );
	}

	/**
	 * {@inheritdoc}
	 */
	public function validate_fields() {
		$is_error_add_card = $this->get_posted_value( 'is_error_add_card' );
		if ( $is_error_add_card ) {
			$error_msg = $this->get_posted_value( 'error_add_card_form' );
			if ( $error_msg ) {
				foreach ( explode( '|', $error_msg ) as $msg ) {
					wc_add_notice( '<strong>' . $msg . '</strong>', 'error' );
				}
			} else {
				wc_add_notice( '<strong>Something went wrong. Refresh to try again.</strong>', 'error' );
			}
			return false;
		}

		$error = WC()->session->get( 'waave_validation_error' );
		if ( $error ) {
				wc_add_notice( '<strong>' . $error . '</strong>', 'error' );
				return false;
		}

		$this->validation_code = WC()->session->get( 'waave_validation_code', '' );
		$this->age_check_id    = $this->get_posted_value( 'age_check_id', '' );

		return true;
	}

	/**
	 * Thank you page.
	 *
	 * @param int $order_id order id.
	 */
	public function thankyou_page( $order_id ) {
		$this->update_order_status( $order_id );
		echo wp_kses( '<p>Thank you for your order with WAAVE.</p>', array( 'p' => array() ) );
	}

	/**
	 * Update order status.
	 *
	 * @param string $order_id order id.
	 */
	public function update_order_status( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( $this->is_successful_order( $order ) ) {
			return;
		}

		$this->log( 'Start update order status: ' . $order_id );

		$args = array(
			'venue_id'     => $this->venue_id,
			'reference_id' => $order_id,
			'amount'       => $order->get_total(),
		);

		$url = $this->api_url . '/complete/transactions/status';

		$response = wp_remote_get( add_query_arg( $args, $url ) );
		$body     = wp_remote_retrieve_body( $response );

		$this->log( 'WAAVE response: ' . $body );

		$result = json_decode( $body, true );
		if ( isset( $result['status'] ) && 'completed' === $result['status'] ) {
			$order->add_order_note( 'WAAVE payment completed' );
			$order->payment_complete();
		}

		$this->log( 'Updated order status: ' . $order_id );
	}

	/**
	 * Function checkout_update_order_review.
	 *
	 * @param string $data data.
	 */
	public static function checkout_update_order_review( $data ) {
		unset( WC()->session->waave_validation_error );
		unset( WC()->session->waave_validation_code );
		unset( WC()->session->waave_age_check_id );
		unset( WC()->session->waave_is_need_age_check );

		parse_str( $data, $output );
		if ( empty( $output ) ) {
			return $data;
		}

		$products = self::get_cart_products();
		$billing  = self::get_billing( $output );
		$shipping = self::get_shipping_from_billing( $billing );

		if ( ! empty( $output['ship_to_different_address'] ) ) {
			$shipping = self::get_shipping( $output );
		}

		if ( empty( $shipping['shipping_city'] ) || empty( $shipping['shipping_state'] ) || empty( $shipping['shipping_country'] ) || empty( $shipping['shipping_postcode'] ) || empty( $shipping['shipping_address_1'] ) ) {
			return $data;
		}

		$venue_id = get_option( 'waave_compliance_venue_id' );
		if ( is_plugin_inactive( 'waave-compliance/waave-compliance.php' ) ) {
			$settings = get_option( 'woocommerce_waave_checkout_settings' );
			$venue_id = $settings['venue_id'];
		}

		$body = array(
			'venue_id'          => $venue_id,
			'amount'            => WC()->cart->get_cart_contents_total(),
			'email'             => $billing['billing_email'],
			'ip_address'        => WC_Geolocation::get_ip_address(),
			'products'          => $products,
			'billing'           => $billing,
			'shipping'          => $shipping,
			'is_need_age_check' => true,
		);

		$base_url = self::API_PROD_URL;
		$settings = get_option( 'woocommerce_waave_checkout_settings' );
		if ( 'yes' === $settings['testmode'] ) {
			$base_url = self::API_SANDBOX_URL;
		}

		$url     = $base_url . '/compliance/validate';
		$options = array(
			'body' => $body,
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		if ( isset( $response['message'] ) && empty( $response['success'] ) ) {
			WC()->session->set( 'waave_validation_error', '<strong>' . $response['message'] . '</strong>' );
		}

		if ( isset( $response['validation_code'] ) ) {
			WC()->session->set( 'waave_validation_code', $response['validation_code'] );
		}

		if ( isset( $response['age_check_id'] ) ) {
			WC()->session->set( 'waave_age_check_id', $response['age_check_id'] );
		}

		if ( isset( $response['is_need_to_check'] ) ) {
			WC()->session->set( 'waave_is_need_age_check', (bool) $response['is_need_to_check'] );
		}

		return $data;
	}

	/**
	 * Function review_order_after_submit.
	 */
	public static function review_order_after_submit() {
		$waave_is_need_age_check = WC()->session->get( 'waave_is_need_age_check', false );
		?>
		<input id='is_need_age_check' name='is_need_age_check' type='hidden' value='<?php echo esc_html( $waave_is_need_age_check ); ?>' />
		<?php
		$waave_age_check_id = WC()->session->get( 'waave_age_check_id', '' );
		if ( $waave_age_check_id ) {
			?>
			<input id='age_check_id' name='age_check_id' type='hidden' value='<?php echo esc_html( $waave_age_check_id ); ?>' />
			<?php
		}
	}

	/**
	 * Function waave_admin_enqueue_scripts.
	 *
	 * @return void
	 */
	public static function waave_admin_enqueue_scripts() {
		$data    = get_plugin_data( WC_WAAVE_MAIN_FILE );
		$version = $data['Version'];

		wp_enqueue_script( 'waave_admin', plugins_url( '../assets/js/admin.js', __FILE__ ), array( 'jquery' ), $version, true );
		wp_enqueue_style( 'waave_admin', plugins_url( '../assets/css/admin.css', __FILE__ ), array(), $version );
	}

	/**
	 * Function add_settings_popup
	 *
	 * @return void
	 */
	public static function add_settings_popup() {
		if ( ! get_option( 'waave_complete_show_setup_popup' ) ) {
			return;
		}
		?>
		<div id="waave-setup-popup">
			<div class="popup-overlay"></div>
			<div class="popup-content">
				<h2>WAAVE Complete</h2>
				<form id="waave-setup-form" method="post">
				<?php wp_nonce_field( 'waave_complete_setup_form_key', 'waave-nonce' ); ?>

					<div id="waave-form-message" style="display: none;"></div>
					<table class="form-table">
						<tr>
							<th scope="row">
								<label for="waave-email">Email <span class="required">*</span></label>
							</th>
							<td>
								<input type="text" id="waave-email" class="regular-text" required>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="waave-password">Password <span class="required">*</span></label>
							</th>
							<td class="password-input">
								<input type="password" id="waave-password" required>
								<span class="show-password-input"></span>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="waave-testmode">Sandbox</label>
							</th>
							<td>
								<input type="checkbox" id="waave-testmode">
							</td>
						</tr>
					</table>
					<div class="submit">
						<button type="submit" id="waave-submit-button" class="button button-primary">Save</button>
						<button type="button" id="waave-close-button" class="button">Close</button>
						<span id="waave-loading-spinner" class="spinner" style="visibility: hidden;"></span>
					</div>
				</form>
			</div>
		</div>
		<?php
		delete_option( 'waave_complete_show_setup_popup' );
	}

	/**
	 * Function save_settings_popup
	 *
	 * @return void
	 */
	public static function save_settings_popup() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ) );
		}

		$is_valid = check_ajax_referer( 'waave_complete_setup_form_key', 'nonce' );

		if ( ! $is_valid ) {
			wp_send_json_error( array( 'message' => 'The nonce is invalid' ) );
		}

		if ( ! isset( $_POST['email'] ) ||
			! isset( $_POST['password'] ) ) {
			wp_send_json_error( array( 'message' => 'Invalid credentials!' ) );
		}

		$email    = sanitize_text_field( wp_unslash( $_POST['email'] ) );
		$password = sanitize_text_field( wp_unslash( $_POST['password'] ) );
		$testmode = isset( $_POST['testmode'] ) && '1' === $_POST['testmode'] ? 'yes' : 'no';

		$response = self::call_waave_verify_api( $email, $password, $testmode );

		if ( isset( $response['message'] ) ) {
			wp_send_json_error( $response );
		}

		if ( empty( $response['venue_id'] ) ||
			empty( $response['access_key'] ) ||
			empty( $response['private_key'] ) ) {
			wp_send_json_error( array( 'message' => 'Invalid credentials!' ) );
		}

		$settings = array(
			'venue_id'    => $response['venue_id'],
			'access_key'  => $response['access_key'],
			'private_key' => $response['private_key'],
			'testmode'    => $testmode,
		);

		update_option( 'woocommerce_waave_checkout_settings', $settings );

		wp_send_json_success( array( 'message' => 'WAAVE Compliance settings save successfully!' ) );
	}

	/**
	 * Get POST data.
	 *
	 * @param string $key key.
	 * @param string $default default value.
	 */
	protected function get_posted_value( $key, $default = '' ) {
		// phpcs:ignore
		if ( empty( $_POST[ $key ] ) ) {
			return $default;
		}

		// phpcs:ignore
		$value = wc_clean( wp_unslash( $_POST[ $key ] ) );

		if ( is_string( $value ) ) {
			return trim( $value );
		}

		return $value;
	}

	/**
	 * Log system processes.
	 *
	 * @since 1.0.0
	 * @param string $message message.
	 */
	protected function log( $message ) {
		if ( empty( $this->logger ) ) {
			$this->logger = new WC_Logger();
		}

		$this->logger->add( 'waave', $message );
	}

	/**
	 * Checks if is order successfully.
	 *
	 * @param Order $order order.
	 * @return boolean
	 */
	protected function is_successful_order( $order ) {
		$status = $order->get_status();
		return in_array( $status, array( 'processing', 'completed' ), true );
	}

	/**
	 * Function call_waave_verify_api
	 *
	 * @param string $email email.
	 * @param string $password password.
	 * @param string $testmode test mode.
	 */
	private static function call_waave_verify_api( $email, $password, $testmode ) {
		$url = self::API_PROD_URL;
		if ( 'yes' === $testmode ) {
			$url = self::API_SANDBOX_URL;
		}

		$url .= '/compliance/verify';

		$options = array(
			'body' => array(
				'email'    => $email,
				'password' => $password,
			),
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		return $response;
	}

	/**
	 * Get cart products.
	 */
	private static function get_cart_products() {
		$products = array();

		$cart_contents = WC()->cart->get_cart();
		foreach ( $cart_contents as $value ) {
			$product = $value['data'];
			if ( empty( $product ) ) {
				continue;
			}

			$product_id = $product->get_id();
			$categories = $product->get_category_ids();
			if ( 'variation' === $product->get_type() ) {
				$product_id = $product->get_parent_id();
				$parent     = wc_get_product( $product_id );
				$categories = $parent->get_category_ids();
			}

			$temp = array(
				'id'               => $product_id,
				'name'             => $product->get_name(),
				'sku'              => $product->get_sku(),
				'price'            => $product->get_price(),
				'quantity'         => $value['quantity'],
				'categories'       => $categories,
				'type'             => '',
				'billing_period'   => '',
				'billing_interval' => '',
			);

			if ( class_exists( 'WC_Subscriptions_Product' ) && WC_Subscriptions_Product::is_subscription( $product ) ) {
				$temp['type']             = $product->get_type();
				$temp['billing_period']   = WC_Subscriptions_Product::get_period( $product );
				$temp['billing_interval'] = WC_Subscriptions_Product::get_interval( $product );
			}

			$products[] = $temp;
		}

		return $products;
	}

	/**
	 * Get billing.
	 *
	 * @param array $output output.
	 */
	private static function get_billing( $output ) {
		$billing_first_name = ! empty( $output['billing_first_name'] ) ? $output['billing_first_name'] : '';
		$billing_last_name  = ! empty( $output['billing_last_name'] ) ? $output['billing_last_name'] : '';
		$billing_country    = ! empty( $output['billing_country'] ) ? $output['billing_country'] : '';
		$billing_address_1  = ! empty( $output['billing_address_1'] ) ? $output['billing_address_1'] : '';
		$billing_address_2  = ! empty( $output['billing_address_2'] ) ? $output['billing_address_2'] : '';
		$billing_city       = ! empty( $output['billing_city'] ) ? $output['billing_city'] : '';
		$billing_state      = ! empty( $output['billing_state'] ) ? $output['billing_state'] : 'Municipality';
		$billing_postcode   = ! empty( $output['billing_postcode'] ) ? $output['billing_postcode'] : '';
		$billing_phone      = ! empty( $output['billing_phone'] ) ? $output['billing_phone'] : '';
		$billing_email      = ! empty( $output['billing_email'] ) ? $output['billing_email'] : '';

		return array(
			'billing_first_name' => trim( $billing_first_name ),
			'billing_last_name'  => trim( $billing_last_name ),
			'billing_country'    => trim( $billing_country ),
			'billing_address_1'  => trim( $billing_address_1 ),
			'billing_address_2'  => trim( $billing_address_2 ),
			'billing_city'       => trim( $billing_city ),
			'billing_state'      => trim( $billing_state ),
			'billing_postcode'   => trim( $billing_postcode ),
			'billing_phone'      => trim( $billing_phone ),
			'billing_email'      => trim( $billing_email ),
		);
	}

	/**
	 * Get shipping.
	 *
	 * @param array $output output.
	 */
	private static function get_shipping( $output ) {
		$shipping_first_name = ! empty( $output['shipping_first_name'] ) ? $output['shipping_first_name'] : '';
		$shipping_last_name  = ! empty( $output['shipping_last_name'] ) ? $output['shipping_last_name'] : '';
		$shipping_country    = ! empty( $output['shipping_country'] ) ? $output['shipping_country'] : '';
		$shipping_address_1  = ! empty( $output['shipping_address_1'] ) ? $output['shipping_address_1'] : '';
		$shipping_address_2  = ! empty( $output['shipping_address_2'] ) ? $output['shipping_address_2'] : '';
		$shipping_city       = ! empty( $output['shipping_city'] ) ? $output['shipping_city'] : '';
		$shipping_state      = ! empty( $output['shipping_state'] ) ? $output['shipping_state'] : 'Municipality';
		$shipping_postcode   = ! empty( $output['shipping_postcode'] ) ? $output['shipping_postcode'] : '';

		return array(
			'shipping_first_name' => trim( $shipping_first_name ),
			'shipping_last_name'  => trim( $shipping_last_name ),
			'shipping_country'    => trim( $shipping_country ),
			'shipping_address_1'  => trim( $shipping_address_1 ),
			'shipping_address_2'  => trim( $shipping_address_2 ),
			'shipping_city'       => trim( $shipping_city ),
			'shipping_state'      => trim( $shipping_state ),
			'shipping_postcode'   => trim( $shipping_postcode ),
		);
	}

	/**
	 * Get shipping from billing.
	 *
	 * @param array $billing billing.
	 */
	private static function get_shipping_from_billing( $billing ) {
		$billing_first_name = ! empty( $billing['billing_first_name'] ) ? $billing['billing_first_name'] : '';
		$billing_last_name  = ! empty( $billing['billing_last_name'] ) ? $billing['billing_last_name'] : '';
		$billing_country    = ! empty( $billing['billing_country'] ) ? $billing['billing_country'] : '';
		$billing_address_1  = ! empty( $billing['billing_address_1'] ) ? $billing['billing_address_1'] : '';
		$billing_address_2  = ! empty( $billing['billing_address_2'] ) ? $billing['billing_address_2'] : '';
		$billing_city       = ! empty( $billing['billing_city'] ) ? $billing['billing_city'] : '';
		$billing_state      = ! empty( $billing['billing_state'] ) ? $billing['billing_state'] : 'Municipality';
		$billing_postcode   = ! empty( $billing['billing_postcode'] ) ? $billing['billing_postcode'] : '';

		return array(
			'shipping_first_name' => trim( $billing_first_name ),
			'shipping_last_name'  => trim( $billing_last_name ),
			'shipping_country'    => trim( $billing_country ),
			'shipping_address_1'  => trim( $billing_address_1 ),
			'shipping_address_2'  => trim( $billing_address_2 ),
			'shipping_city'       => trim( $billing_city ),
			'shipping_state'      => trim( $billing_state ),
			'shipping_postcode'   => trim( $billing_postcode ),
		);
	}
}
