<?php
/**
 * Plugin Name: WAAVE Compliance
 * Description: WAAVE Compliance
 * Author: WAAVE
 * Author URI: https://getwaave.com/
 * Version: 1.2.6
 * Requires at least: 4.6
 * WC requires at least: 4.0
 * WC tested up to: 5.0
 * Text Domain: waave-compliance
 * Domain Path: /i18n/languages/
 *
 * @package WAAVE_Compliance
 */

defined( 'ABSPATH' ) || exit;

define( 'WAAVE_COMPLIANCE_PLUGIN_URL', untrailingslashit( plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) ) ) );
define( 'WAAVE_COMPLIANCE_PLUGIN_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
define( 'WAAVE_COMPLIANCE_MAIN_FILE', __FILE__ );

// Make sure WooCommerce is active.
if ( ! in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ), true ) ) {
	return;
}

register_activation_hook(
	__FILE__,
	function() {
		add_option( 'waave_compliance_show_setup_popup', true );
	}
);

register_deactivation_hook(
	__FILE__,
	array( 'WAAVE_Compliance', 'plugin_deactivation' )
);

/**
 * WAAVE Compliance init
 */
function waave_compliance_init() {
	require_once plugin_basename( 'vendor/autoload.php' );
	require_once plugin_basename( 'includes/class-waave-compliance.php' );
	require_once plugin_basename( 'includes/class-waave-compliance-rest-api.php' );
	require_once plugin_basename( 'includes/class-waave-compliance-age-gate.php' );
}
add_action( 'plugins_loaded', 'waave_compliance_init' );

if ( is_admin() ) {
	add_action( 'init', array( 'WAAVE_Compliance', 'init' ) );
}
add_action( 'init', array( 'WAAVE_Compliance_Age_Gate', 'init' ) );
add_action( 'rest_api_init', array( 'WAAVE_Compliance_REST_API', 'init' ) );

add_action(
	'before_woocommerce_init',
	function() {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
		}
	}
);

// ==========================================
// Autoupdate function start here
// Will be removed in the WP store version
// ==========================================

/**
 * WAAVE Compliance update
 *
 * @param object $transient transient.
 */
function waave_compliance_update( $transient ) {
	if ( empty( $transient->checked ) ) {
		return $transient;
	}

	$remote = waave_compliance_request();

	$data    = get_plugin_data( WAAVE_COMPLIANCE_MAIN_FILE );
	$version = $data['Version'];

	if (
		$remote
		&& version_compare( $version, $remote->version, '<' )
		&& version_compare( $remote->requires, get_bloginfo( 'version' ), '<' )
		&& version_compare( $remote->requires_php, PHP_VERSION, '<' )
	) {
		$res              = new stdClass();
		$res->slug        = plugin_basename( __DIR__ );
		$res->plugin      = plugin_basename( __FILE__ );
		$res->new_version = $remote->version;
		$res->tested      = $remote->tested;
		$res->package     = $remote->download_url;

		$transient->response[ $res->plugin ] = $res;
	}

	return $transient;
}
add_filter( 'site_transient_update_plugins', 'waave_compliance_update' );

/**
 * WAAVE Compliance request
 */
function waave_compliance_request() {
	$cache_key = 'waave_compliance_update_key';
	$remote    = get_transient( $cache_key );

	$url = 'https://getwaave.co/uploads/plugins/compliance/info.json';
	if ( 'yes' === get_option( 'waave_compliance_testmode' ) ) {
		$url = 'https://staging.getwaave.co/uploads/plugins/compliance/info.json';
	}

	if ( false === $remote ) {
		$remote = wp_remote_get(
			$url,
			array(
				'timeout' => 10,
				'headers' => array(
					'Accept' => 'application/json',
				),
			)
		);

		if (
		is_wp_error( $remote )
		|| 200 !== wp_remote_retrieve_response_code( $remote )
		|| empty( wp_remote_retrieve_body( $remote ) )
		) {
			return false;
		}

		set_transient( $cache_key, $remote, DAY_IN_SECONDS );
	}

	$remote = json_decode( wp_remote_retrieve_body( $remote ) );

	return $remote;
}

/**
 * WAAVE Compliance info
 *
 * @param string $res res.
 * @param string $action action.
 * @param array  $args args.
 */
function waave_compliance_info( $res, $action, $args ) {
	if ( 'plugin_information' !== $action ) {
		return $res;
	}

	if ( plugin_basename( __DIR__ ) !== $args->slug ) {
		return $res;
	}

	// get updates.
	$remote = waave_compliance_request();

	if ( ! $remote ) {
		return $res;
	}

	$res = new stdClass();

	$res->name           = $remote->name;
	$res->slug           = $remote->slug;
	$res->version        = $remote->version;
	$res->tested         = $remote->tested;
	$res->requires       = $remote->requires;
	$res->author         = $remote->author;
	$res->author_profile = $remote->author_profile;
	$res->download_link  = $remote->download_url;
	$res->trunk          = $remote->download_url;
	$res->requires_php   = $remote->requires_php;
	$res->last_updated   = $remote->last_updated;

	$res->sections = array(
		'description'  => $remote->sections->description,
		'installation' => $remote->sections->installation,
		'changelog'    => $remote->sections->changelog,
	);

	if ( ! empty( $remote->banners ) ) {
		$res->banners = array(
			'low'  => $remote->banners->low,
			'high' => $remote->banners->high,
		);
	}

	return $res;
}
add_filter( 'plugins_api', 'waave_compliance_info', 20, 3 );

/**
 * WAAVE Compliance purge
 */
function waave_compliance_purge() {
	delete_transient( 'waave_compliance_update_key' );
}
add_action( 'upgrader_process_complete', 'waave_compliance_purge', 10 );
