<?php
/**
 * WAAVE Credit Card Direct
 *
 * We load it later to ensure WC is loaded first since we're extending it.
 *
 * @class       WC_WAAVE_Credit_Card_Direct
 * @extends     WC_WAAVE_Complete
 * @package     WAAVE Complete
 */

/**
 * WC_WAAVE_Credit_Card_Direct
 */
class WC_WAAVE_Credit_Card_Direct extends WC_WAAVE_Complete {

	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {
		parent::__construct();

		$this->id                 = 'waave_guest_credit_card';
		$this->method_title       = 'WAAVE Complete';
		$this->method_description = 'WAAVE guest credit card direct.';

		// Common fields.
		$title     = 'Credit and Debit Cards Payment';
		$incognito = get_option( 'waave_incognito_enabled' );
		if ( 'yes' !== $incognito ) {
			$title .= ' - powered by WAAVE';
		}
		$this->title = $title;

		$this->has_fields = true;

		// Subscriptions.
		$this->supports = array(
			'products',
			'subscriptions',
			'subscription_suspension',
			'subscription_cancellation',
			'subscription_reactivation',
			'subscription_amount_changes',
			'subscription_date_changes',
			'multiple_subscriptions',
		);

		// Actions.
		add_action( 'woocommerce_thankyou_waave_guest_credit_card', array( $this, 'thankyou_page' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'plugin_assets' ) );
		add_action( 'woocommerce_after_checkout_form', array( $this, 'flexipay_add_forms' ) );

		if ( class_exists( 'WC_Subscriptions_Product' ) ) {
			add_action( 'woocommerce_scheduled_subscription_payment', array( $this, 'scheduled_subscription_payment' ) );
			add_action( 'woocommerce_api_wc_waave_credit_card_direct', array( $this, 'cancel_subscription' ) );
			add_action( 'woocommerce_subscription_cancelled_waave_guest_credit_card', array( $this, 'do_cancelled_subscription' ) );
		}
	}

	/**
	 * Function plugin_assets.
	 */
	public function plugin_assets() {
		$plugin_data = get_plugin_data( WC_WAAVE_MAIN_FILE );
		wp_enqueue_style( 'woocommerce_waave_style', plugins_url( 'assets/css/style.css', WC_WAAVE_MAIN_FILE ), array(), $plugin_data['Version'] );
	}

	/**
	 * {@inheritdoc}
	 */
	public function payment_fields() {
		?>

		<fieldset id="wc-<?php echo esc_attr( $this->id ); ?>-cc-form" class='wc-credit-card-form wc-payment-form'>
			<?php do_action( 'woocommerce_credit_card_form_start', $this->id ); ?>
				<form id="frmAddCard">
					<div class="card-body py-3">
						<p class="small alert alert-danger" style="display: none"></p>
						<p class="form-row form-row-wide form-row-wide">
							<strong>Enter card information</strong>
						</p>
						<div class="form-group">
							<div class="input-group">
								<label for="">Card number&nbsp;<span class="required">*</span></label>
								<div id="cc-number" class="form-field"></div>
							</div>
						</div>
						<div class="form-group form-row-first">
							<div class="input-group">
								<label for="">Expiration (MM/YY)&nbsp;<span class="required">*</span></label>
								<div id="cc-expiration-date" class="form-field"></div>
							</div>
						</div>
						<div class="form-group form-row-last">
							<div class="input-group">
								<label for="">Card Security Code&nbsp;<span class="required">*</span></label>
								<div id="cc-cvc" class="form-field"></div>
							</div>
						</div>
					</div>
					<div id="venue_id" style="display: none;"></div>
				</form>
				<?php do_action( 'woocommerce_credit_card_form_end', $this->id ); ?>
			<div class="clear"></div>
		</fieldset>
		<?php
	}

	/**
	 * {@inheritdoc}
	 *
	 * @param string $order_id order id.
	 * @throws Exception Exception.
	 */
	public function process_payment( $order_id ) {
		$order        = wc_get_order( $order_id );
		$data_to_send = $this->prepare_data( $order );

		$card_token = $this->get_posted_value( $this->id . '-card-token' );
		if ( '' === $card_token ) {
			throw new Exception( 'Invalid Card or Billing Address info! Please re-check and try again!' );
		}

		$data_to_send['card_information'] = array(
			'card_token' => $card_token,
		);

		$data_to_send['shipping_cost'] = $order->get_shipping_total();

		$validation_code = $this->validation_code;
		if ( ! $validation_code ) {
			$validation_code = $this->get_validation_code( $order );
		}
		$data_to_send['validation_code'] = $validation_code;

		if ( class_exists( 'WC_Subscriptions_Product' ) && wcs_order_contains_subscription( $order ) ) {
			$order->add_meta_data( 'card_token', $card_token );
			$order->save_meta_data();
		}

		// Card #764, 3ds.
		$enroll_data = $this->get_posted_value( 'enroll_data' );
		if ( $enroll_data ) {
			$order->update_meta_data( '_custom_3ds_data', $data_to_send );
			$order->save();

			$data_to_send['device_information']    = json_decode( $enroll_data, true );
			$data_to_send['flexipay_access_token'] = $this->get_posted_value( 'flexipay_access_token' );

			$response = $this->flexipay_enroll( $data_to_send );

			if ( isset( $response['message'] ) && empty( $response['success'] ) ) {
				throw new Exception( $response['message'] );
			}

			if ( isset( $response['flexipay_access_token'] ) && isset( $response['flexipay_stepup_url'] ) ) {
				return array(
					'result'       => 'success',
					'messages'     => '<ul></ul>',
					'access_token' => $response['flexipay_access_token'],
					'stepup_url'   => $response['flexipay_stepup_url'],
					'flexipay_md'  => $response['flexipay_md'],
					'order_id'     => $order_id,
				);
			}
		}

		set_time_limit( 0 );
		$this->make_transaction( $data_to_send );

		wc_empty_cart();

		return array(
			'result'   => 'success',
			'redirect' => $this->get_return_url( $order ),
		);
	}

	/**
	 * API cancel subscription.
	 */
	public function cancel_subscription() {
		$json = file_get_contents( 'php://input' );
		$data = json_decode( $json, true );
		$data = stripslashes_deep( $data );

		ob_end_clean();

		$body = json_encode( $data ); // phpcs:ignore
		$this->log( 'Cancel Subscription Data: ' . $body );

		$url = add_query_arg(
			array(
				'wc-api' => 'WC_WAAVE_Credit_Card_Direct',
			),
			home_url( '/' )
		);

		$signature        = hash( 'sha256', $this->private_key . $url . $body );
		$header_signature = isset( $_SERVER['HTTP_X_API_SIGNATURE'] ) ? wp_unslash($_SERVER['HTTP_X_API_SIGNATURE']) : ''; // phpcs:ignore

		if ( $signature !== $header_signature ) {
			$this->log( 'Signature is invalid.' );
			$this->log( 'Callback url: ' . $url );

			wp_send_json(
				array(
					'success' => false,
					'message' => 'Signature is invalid',
				)
			);
		}

		$subscription_id = absint( $data['reference_id'] );

		$subscription = wcs_get_subscription( $subscription_id );
		$subscription->cancel_order( 'WAAVE update status to cancelled.' );

		$this->log( 'Successfully cancel subscription.' );

		wp_send_json( array( 'success' => true ) );
	}

	/**
	 * Action subscriptions_cancelled_for_order.
	 *
	 * @param Subscription $subscription subscription.
	 */
	public function do_cancelled_subscription( $subscription ) {
		$url = $this->api_url . '/subscriptions/cancel';

		$data = array(
			'venue_id'     => (string) $this->venue_id,
			'reference_id' => (string) $subscription->get_id(),
		);

		$body      = json_encode( $data ); // phpcs:ignore
		$signature = hash( 'sha256', $this->private_key . $url . $body );

		$this->log( 'Call subscription cancel API.' );
		$this->log( 'Url: ' . $url );
		$this->log( 'Body: ' . $body );

		$options = array(
			'body'    => $data,
			'timeout' => 300,
			'headers' => array(
				'X-Api-Signature' => $signature,
			),
		);

		$request = wp_remote_post( $url, $options );
		$this->log( 'Response: ' . wp_remote_retrieve_body( $request ) );
	}

	/**
	 * Renewal subscription.
	 *
	 * @param string $subscription_id subscription_id.
	 */
	public function scheduled_subscription_payment( $subscription_id ) {
		$subscription = wcs_get_subscription( $subscription_id );

		$order        = $subscription->get_last_order( 'all' );
		$data_to_send = $this->prepare_data( $order, true );

		$parent_order = $subscription->get_parent();
		$card_token   = $parent_order->get_meta( 'card_token' );

		$data_to_send['card_information'] = array(
			'card_token' => $card_token,
		);

		$data_to_send['shipping_cost'] = $order->get_shipping_total();

		$this->log( 'RENEWAL DATA: ' . wp_json_encode( $data_to_send ) );

		$this->make_transaction( $data_to_send, true );
	}

	/**
	 * Subscription expired.
	 *
	 * @param Subscription $subscription subscription.
	 */
	public static function subscription_status_expired( $subscription ) {
		$settings = get_option( 'woocommerce_waave_checkout_settings' );

		$base_url = self::API_PROD_URL;
		if ( 'yes' === $settings['testmode'] ) {
			$base_url = self::API_SANDBOX_URL;
		}
		$url = $base_url . '/subscriptions/status';

		$venue_id = get_option( 'waave_compliance_venue_id' );
		if ( is_plugin_inactive( 'waave-compliance/waave-compliance.php' ) ) {
			$venue_id = $settings['venue_id'];
		}

		$data = array(
			'venue_id'     => (string) $venue_id,
			'reference_id' => (string) $subscription->get_id(),
			'status'       => 'expired',
		);

		$body      = json_encode( $data ); // phpcs:ignore
		$signature = hash( 'sha256', $settings['private_key'] . $url . $body );

		$logger = new WC_Logger();
		$logger->add( 'waave', 'Call subscription status API.' );
		$logger->add( 'waave', 'Url: ' . $url );
		$logger->add( 'waave', 'Body: ' . $body );

		$options = array(
			'body'    => $data,
			'timeout' => 300,
			'headers' => array(
				'X-Api-Signature' => $signature,
			),
		);

		$request = wp_remote_post( $url, $options );
		$logger->add( 'waave', 'Response: ' . wp_remote_retrieve_body( $request ) );
	}

	/**
	 * Function continue_payment.
	 * Card #764, 3DS
	 *
	 * @throws Exception Exception.
	 */
	public static function flexipay_continue_payment() {
		$order_id = wc_clean( wp_unslash( $_POST['order_id'] ) ); //phpcs:ignore
		$order    = wc_get_order( $order_id );

		if ( empty( $order ) ) {
			wp_send_json(
				array(
					'result'  => false,
					'message' => 'Order not found!',
				)
			);
		}

		$settings = get_option( 'woocommerce_waave_checkout_settings' );

		$base_url = self::API_TRANS_PROD_URL;
		if ( 'yes' === $settings['testmode'] ) {
			$base_url = self::API_TRANS_SANDBOX_URL;
		}
		$url = $base_url . '/transactions';

		$data = $order->get_meta( '_custom_3ds_data' );

		//phpcs:ignore
		$parseDsThirdParties = wc_clean( wp_unslash( $_POST['parseDsThirdParties'] ) );
		$data['parse_ds_third_parties'] = $parseDsThirdParties;

		$body = json_encode( $data ); // phpcs:ignore

		$signature = hash( 'sha256', $settings['private_key'] . $url . $body );

		$options = array(
			'body'    => $data,
			'timeout' => 300,
			'headers' => array(
				'X-Api-Signature' => $signature,
			),
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		if ( isset( $response['message'] ) && empty( $response['success'] ) ) {
			wp_send_json(
				array(
					'result'  => false,
					'message' => $response['message'],
				)
			);
		}

		wc_empty_cart();

		wp_send_json(
			array(
				'result'   => 'success',
				'redirect' => $order->get_checkout_order_received_url(),
			)
		);
	}

	/**
	 * Flexipay form.
	 */
	public function flexipay_add_forms() {
		?>
		<iframe name="collection-iframe" height="10" width="10" style="display: none;"></iframe>
		<form id="collection-form" method="POST" target="collection-iframe">
			<input id="collection-form-input" type="hidden" name="JWT">
		</form>

		<div id="step-up-iframe-overlay" class="step-up-iframe-overlay">
			<div class="step-up-iframe-container">
				<iframe name="step-up-iframe"></iframe>
				<form id="step-up-form" target="step-up-iframe" method="post">
					<input type="hidden" name="JWT" id="step-up-form-jwt"/>
					<input type="hidden" name="MD" id="step-up-form-md"/>
				</form>
			</div>
		</div>

		<?php
	}

	/**
	 * Function flexipay_enroll.
	 * Card #764, 3DS
	 *
	 * @param array $enroll_body enroll body.
	 */
	private function flexipay_enroll( $enroll_body ) {
		$url = $this->api_trans_url . '/cards/enroll';

		$options = array(
			'body' => $enroll_body,
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		return $response;
	}

	/**
	 * Prepare data.
	 *
	 * @param Order $order order.
	 * @param bool  $is_renewal is renewal.
	 * @return array
	 */
	private function prepare_data( $order, $is_renewal = false ) {
		$data_to_send = array(
			// WAAVE details.
			'access_key'   => $this->access_key,
			'venue_id'     => $this->venue_id,
			'return_url'   => $this->get_return_url( $order ),
			'callback_url' => $this->callback_url,
			'cancel_url'   => $order->get_cancel_order_url(),

			// Order details.
			'amount'       => (string) $order->get_total(),
			'reference_id' => (string) $order->get_id(),
			'currency'     => $order->get_currency(),
			'customer_ip'  => $order->get_customer_ip_address(),
		);

		$data_to_send['age_check_id'] = $this->get_posted_value( 'age_check_id' );

		$order_data = $order->get_data();
		$billing    = isset( $order_data['billing'] ) ? $order_data['billing'] : array();
		if ( $billing ) {
			$billing['phone_code'] = WC()->countries->get_country_calling_code( $billing['country'] );
		}
		if ( empty( $billing['state'] ) ) {
			$billing['state'] = 'Municipality';
		}

		$shipping                  = $billing;
		$ship_to_different_address = $this->get_posted_value( 'ship_to_different_address', 0 );
		if ( $ship_to_different_address && isset( $order_data['shipping'] ) ) {
			$shipping = $order_data['shipping'];
			if ( $shipping ) {
				$shipping['phone_code'] = WC()->countries->get_country_calling_code( $shipping['country'] );
			}
		}
		if ( empty( $shipping['state'] ) ) {
			$shipping['state'] = 'Municipality';
		}

		$data_to_send['billing']  = $billing;
		$data_to_send['shipping'] = $shipping;

		if ( ! class_exists( 'WC_Subscriptions_Product' ) ) {
			return $data_to_send;
		}

		$order_type = 'parent';
		if ( $is_renewal ) {
			$order_type = 'renewal';
		}

		if ( wcs_order_contains_subscription( $order, $order_type ) ) {
			$order_type = 'parent';
			if ( $is_renewal ) {
				$order_type = 'renewal';
			}

			$list          = wcs_get_subscriptions_for_order( $order, array( 'order_type' => $order_type ) );
			$subscriptions = array();
			foreach ( $list as $subscription ) {
				$items = array();
				foreach ( $subscription->get_items() as $value ) {
					$items[] = array(
						'id'       => (string) $value->get_product_id(),
						'name'     => $value->get_name(),
						'quantity' => (string) $value->get_quantity(),
					);
				}

				$subscription = array(
					'reference_id'     => (string) $subscription->get_id(),
					'billing_period'   => $subscription->get_billing_period(),
					'billing_interval' => $subscription->get_billing_interval(),
					'price'            => (string) $subscription->get_total(),
					'trial_end'        => (string) $subscription->get_date( 'trial_end' ),
					'next_payment'     => $subscription->get_date( 'next_payment' ),
					'items'            => $items,
				);

				$subscriptions[] = $subscription;
			}

			$data_to_send['subscriptions'] = $subscriptions;

			$data_to_send['subscription_cancel_url'] = add_query_arg(
				array(
					'wc-api' => 'WC_WAAVE_Credit_Card_Direct',
				),
				home_url( '/' )
			);
		}

		return $data_to_send;
	}

	/**
	 * Make transaction.
	 *
	 * @param array $data_to_send data to send.
	 * @param bool  $is_renewal is renewal.
	 * @throws Exception Exception.
	 * @return array
	 */
	private function make_transaction( $data_to_send, $is_renewal = false ) {
		$url = $this->api_trans_url . '/transactions';

		$body      = json_encode( $data_to_send ); // phpcs:ignore
		$signature = hash( 'sha256', $this->private_key . $url . $body );

		$options = array(
			'body'    => $data_to_send,
			'timeout' => 300,
			'headers' => array(
				'X-Api-Signature' => $signature,
			),
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		if ( isset( $response['message'] ) && empty( $response['success'] ) ) {
			if ( $is_renewal ) {
				$this->log( 'RENEWAL ERROR: ' . $response['message'] );
			} else {
				throw new Exception( $response['message'] );
			}
		}

		if ( isset( $response['status'] ) && 'completed' === $response['status'] ) {
			$order = wc_get_order( $data_to_send['reference_id'] );
			$order->add_order_note( 'WAAVE payment completed' );
			$order->payment_complete();
			$this->log( 'Successfully updated order status: ' . $order->get_id() );
		}

		return true;
	}

	/**
	 * Get validation code again
	 *
	 * @param Order $order Order.
	 * @throws Exception Exception.
	 * @return string
	 */
	private function get_validation_code( $order ) {

		$products = $this->get_order_products( $order );
		$billing  = $this->get_order_billing( $order );
		$shipping = $this->get_order_shipping( $order );

		$body = array(
			'venue_id'          => $this->venue_id,
			'amount'            => WC()->cart->get_cart_contents_total(),
			'email'             => $billing['billing_email'],
			'ip_address'        => $order->get_customer_ip_address(),
			'products'          => $products,
			'billing'           => $billing,
			'shipping'          => $shipping,
			'is_need_age_check' => true,
		);

		$url     = $this->api_url . '/compliance/validate';
		$options = array(
			'body' => $body,
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );

		if ( isset( $response['message'] ) && empty( $response['success'] ) ) {
			throw new Exception( $response['message'] );
		}

		if ( empty( $response['validation_code'] ) ) {
			throw new Exception( 'We were unable to process your order, please try again.' );
		}

		return $response['validation_code'];
	}

	/**
	 * Get cart products.
	 *
	 * @param Order $order Order.
	 */
	private function get_order_products( $order ) {
		$products = array();

		foreach ( $order->get_items() as $item ) {
			$product = $item->get_product();
			if ( empty( $product ) ) {
				continue;
			}

			$product_id = $product->get_id();
			$categories = $product->get_category_ids();
			if ( 'variation' === $product->get_type() ) {
				$product_id = $product->get_parent_id();
				$parent     = wc_get_product( $product_id );
				$categories = $parent->get_category_ids();
			}

			$temp = array(
				'id'         => $product_id,
				'name'       => $product->get_name(),
				'sku'        => $product->get_sku(),
				'price'      => $product->get_price(),
				'quantity'   => $item->get_quantity(),
				'categories' => $categories,
			);

			$products[] = $temp;
		}

		return $products;
	}

	/**
	 * Get billing address
	 *
	 * @param Order $order Order.
	 */
	private function get_order_billing( $order ) {
		return array(
			'billing_first_name' => $order->get_billing_first_name(),
			'billing_last_name'  => $order->get_billing_last_name(),
			'billing_email'      => $order->get_billing_email(),
			'billing_country'    => $order->get_billing_country(),
			'billing_address_1'  => $order->get_billing_address_1(),
			'billing_address_2'  => $order->get_billing_address_2(),
			'billing_city'       => $order->get_billing_city(),
			'billing_state'      => $order->get_billing_state(),
			'billing_postcode'   => $order->get_billing_postcode(),
			'billing_phone'      => $order->get_billing_phone(),
		);
	}

	/**
	 * Get shipping address
	 *
	 * @param Order $order Order.
	 */
	private function get_order_shipping( $order ) {
		return array(
			'shipping_first_name' => $order->get_shipping_first_name(),
			'shipping_last_name'  => $order->get_shipping_last_name(),
			'shipping_country'    => $order->get_shipping_country(),
			'shipping_address_1'  => $order->get_shipping_address_1(),
			'shipping_address_2'  => $order->get_shipping_address_2(),
			'shipping_city'       => $order->get_shipping_city(),
			'shipping_state'      => $order->get_shipping_state(),
			'shipping_postcode'   => $order->get_shipping_postcode(),
		);
	}
}
