<?php
/**
 * Plugin Name: WAAVE Complete
 * Description: WAAVE Checkout and WAAVE guest credit card payment.
 * Author: WAAVE
 * Author URI: https://getwaave.com/
 * Version: 5.7.15
 * Requires at least: 4.6
 * WC requires at least: 4.0
 * WC tested up to: 8.2
 *
 * @package WAAVE Complete
 */

defined( 'ABSPATH' ) || exit;

define( 'WC_WAAVE_PLUGIN_URL', untrailingslashit( plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) ) ) );
define( 'WC_WAAVE_PLUGIN_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
define( 'WC_WAAVE_MAIN_FILE', __FILE__ );
define( 'WAAVE_COMPLETE_PLUGIN_URL', untrailingslashit( plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) ) ) );

register_activation_hook(
	__FILE__,
	function() {
		add_option( 'waave_complete_show_setup_popup', true );
	}
);

/**
 * Declare plugin compatibility with WooCommerce HPOS.
 */
add_action(
	'before_woocommerce_init',
	function() {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
		}
	}
);

/**
 * Initialize the gateway.
 *
 * @since 1.0.0
 */
function wc_waave_complete_init() {
	if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
		return;
	}

	require_once ABSPATH . 'wp-admin/includes/plugin.php';
	require_once plugin_basename( 'includes/class-wc-waave-complete.php' );
	require_once plugin_basename( 'includes/class-wc-waave-checkout.php' );
	require_once plugin_basename( 'includes/class-wc-waave-credit-card-direct.php' );
}
add_action( 'plugins_loaded', 'wc_waave_complete_init' );

/**
 * Add the gateway to WC Available Gateways.
 *
 * @since 1.0.0
 * @param array $gateways all available WC gateways.
 * @return array $gateways all WC gateways + waave complete
 */
function wc_add_waave_complete_to_gateways( $gateways ) {
	$gateways[] = 'WC_WAAVE_Checkout';
	$gateways[] = 'WC_WAAVE_Credit_Card_Direct';
	return $gateways;
}
add_filter( 'woocommerce_payment_gateways', 'wc_add_waave_complete_to_gateways' );


/**
 * Adds plugin page links.
 *
 * @since 1.0.0
 * @param array $links all plugin links.
 * @return array $links all plugin links + our custom links (i.e., "Settings")
 */
function wc_waave_complete_plugin_links( $links ) {

	$plugin_links = array(
		'<a href="' . admin_url( 'admin.php?page=wc-settings&tab=checkout&section=waave_checkout' ) . '">Configure</a>',
	);

	return array_merge( $plugin_links, $links );
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'wc_waave_complete_plugin_links' );

/**
 * Function waave_set_checkout_field_input_value_default.
 *
 * @param array $fields fields.
 */
function waave_set_checkout_field_input_value_default( $fields ) {
	$fields['billing']['billing_country']['default'] = 'Uruguay';
	return $fields;
}
add_filter( 'woocommerce_checkout_fields', 'waave_set_checkout_field_input_value_default' );

/**
 * Function register_waave_age_check_validate_route.
 */
function register_waave_age_check_validate_route() {
	register_rest_route(
		'waave-complete/v1',
		'/age-check-validate',
		array(
			'methods'             => WP_REST_Server::CREATABLE,
			'callback'            => 'age_check_validate_fields',
			'permission_callback' => '__return_true',
		)
	);

	register_rest_route(
		'waave-complete/v1',
		'waavecheckout-state/change',
		array(
			'methods'             => WP_REST_Server::CREATABLE,
			'callback'            => array( 'WC_WAAVE_Checkout', 'change_waave_checkout_state' ),
			'permission_callback' => '__return_true',
		)
	);
}
add_action( 'rest_api_init', 'register_waave_age_check_validate_route' );

/**
 * Function age_check_validate_fields.
 *
 * @param WP_REST_Request $request request.
 */
function age_check_validate_fields( WP_REST_Request $request ) {
	$products = array();
	$params   = $request->get_params();
	if ( $params ) {
		$products = $params['products'];
		$shipping = array(
			'shipping_first_name' => $params['first_name'],
			'shipping_last_name'  => $params['last_name'],
			'shipping_country'    => $params['shipping_country'],
			'shipping_address'    => $params['shipping_address'],
			'shipping_state'      => $params['shipping_state'],
			'shipping_city'       => $params['shipping_city'],
			'shipping_postcode'   => $params['zip'],
			'validation_code'     => WC()->session->get( 'waave_validation_code', '' ),
		);

		$venue_id = get_option( 'waave_compliance_venue_id' );
		if ( is_plugin_inactive( 'waave-compliance/waave-compliance.php' ) ) {
			$venue_id = get_option( 'venue_id' );
		}
		$body    = array(
			'venue_id' => $venue_id,
			'email'    => $params['email'],
			'products' => json_decode( $products ),
			'shipping' => $shipping,
		);
		$url     = $params['url'] . '/compliance/age-check-validate';
		$options = array(
			'body' => $body,
		);

		$request  = wp_remote_post( $url, $options );
		$response = json_decode( wp_remote_retrieve_body( $request ), true );
		return $response;
	}

	WC()->session->set( 'custom_error', 'Error message goes here' );
	return new WP_Error( 'missing_param', 'Something errors. Please refresh and try again.', array( 'status' => 400 ) );
}

/**
 * Function age_check_dob_popup.
 */
function age_check_dob_popup() {
	if ( is_checkout() ) {
		?>
		<div class='age-check-dob-overlay'>
			<div class='age-check-dob-popup'>
			<img class='age-verification-img' src='<?php echo esc_url( WAAVE_COMPLETE_PLUGIN_URL . '/assets/images/logo.png' ); ?>' alt='WAAVE Logo'>
				<h3 class='age-check-dob-title'>Enter your date of birth!</h3>
				<p class='age-check-dob-q'>
					<div class='date-input'>
						<select class='date-input-field' id='age_check_dob_month' required>
						</select>
						<select class='date-input-field' id='age_check_dob_day' required>
						</select>
						<select class='date-input-field' id='age_check_dob_year' required>
						</select>
					</div>
				</p>
				<p>Please enter your date of birth exactly as it appears on your ID mm/dd/yyyy
					<br/> otherwise we would not be able to verify your age.</p>
				<p>
					<a class='age-check-dob-btn age-check-dob-btn-yes' href='javascript:void(0)'>Submit</a>
					<a class='age-check-dob-btn age-check-dob-btn-no' href='javascript:void(0)'>Cancel</a>
				</p>
			</div>
		</div>
		<?php
	}
}
add_action( 'wp_footer', 'age_check_dob_popup' );

/**
 * Function age_check_dob_assets.
 */
function age_check_dob_assets() {
	$data    = get_plugin_data( WC_WAAVE_MAIN_FILE );
	$version = $data['Version'];

	wp_enqueue_script( 'age-check-dob', plugins_url( 'assets/js/age_check_dob.js', __FILE__ ), array( 'jquery' ), $version, true );
	wp_enqueue_style( 'age-check-dob', plugins_url( 'assets/css/age_check_dob.css', __FILE__ ), array(), $version );
}
add_action( 'wp_enqueue_scripts', 'age_check_dob_assets' );

/**
 * Function wc_waave_complete_add_text_age_check.
 */
function wc_waave_complete_add_text_age_check() {
	?>
	<p>Please enter your correct Billing information if it is different from your Shipping address.</p>
	<p>The purchase of some of the products on this site might require ID verification to comply with Federal and State Regulations.</p>
	<?php
}
add_action( 'woocommerce_after_order_notes', 'wc_waave_complete_add_text_age_check' );

/**
 * Function subscription_status_expired.
 *
 * @param object $subscription subscription.
 */
function subscription_status_expired( $subscription ) {
	if ( ! class_exists( 'WC_Subscriptions_Product' ) ) {
		return;
	}

	WC_WAAVE_Credit_Card_Direct::subscription_status_expired( $subscription );
}
add_action( 'woocommerce_subscription_status_expired', 'subscription_status_expired', 10, 2 );

/**
 * Action update order review.
 */
add_action( 'woocommerce_checkout_update_order_review', array( 'WC_WAAVE_Complete', 'checkout_update_order_review' ), 10 );

/**
 * Action review_order_after_submit.
 */
add_action( 'woocommerce_review_order_after_submit', array( 'WC_WAAVE_Complete', 'review_order_after_submit' ), 10 );

/**
 * Action admin_enqueue_scripts.
 */
add_action( 'admin_enqueue_scripts', array( 'WC_WAAVE_Complete', 'waave_admin_enqueue_scripts' ) );

/**
 * Action add settings popup.
 */
add_action( 'admin_footer', array( 'WC_WAAVE_Complete', 'add_settings_popup' ) );

/**
 * Action save settings popup.
 */
add_action( 'wp_ajax_waave_complete_save_settings_popup', array( 'WC_WAAVE_Complete', 'save_settings_popup' ) );

/**
 * Action flexipay_continue_payment.
 */
add_action( 'wc_ajax_waave_complete_flexipay_continue_payment', array( 'WC_WAAVE_Credit_Card_Direct', 'flexipay_continue_payment' ) );

// ==========================================
// Autoupdate function start here
// Put your new code above this lines
// ==========================================

/**
 * Function wc_waave_complete_update.
 *
 * @param object $transient transient.
 */
function wc_waave_complete_update( $transient ) {
	if ( empty( $transient->checked ) ) {
		return $transient;
	}

	$remote = wc_waave_complete_request();

	$data    = get_plugin_data( WC_WAAVE_MAIN_FILE );
	$version = $data['Version'];

	if (
		$remote
		&& version_compare( $version, $remote->version, '<' )
		&& version_compare( $remote->requires, get_bloginfo( 'version' ), '<' )
		&& version_compare( $remote->requires_php, PHP_VERSION, '<' )
	) {
		$res              = new stdClass();
		$res->slug        = plugin_basename( __DIR__ );
		$res->plugin      = plugin_basename( __FILE__ );
		$res->new_version = $remote->version;
		$res->tested      = $remote->tested;
		$res->package     = $remote->download_url;

		$transient->response[ $res->plugin ] = $res;
	}

	return $transient;
}
add_filter( 'site_transient_update_plugins', 'wc_waave_complete_update' );

/**
 * Function wc_waave_complete_request.
 */
function wc_waave_complete_request() {
	$cache_key = 'waave_complete_update_key';
	$remote    = get_transient( $cache_key );

	$settings = get_option( 'woocommerce_waave_checkout_settings' );

	$url = 'https://getwaave.co/uploads/plugins/complete/info.json';
	if ( 'yes' === $settings['testmode'] ) {
		$url = 'https://staging.getwaave.co/uploads/plugins/complete/info.json';
	}

	if ( false === $remote ) {
		$remote = wp_remote_get(
			$url,
			array(
				'timeout' => 10,
				'headers' => array(
					'Accept' => 'application/json',
				),
			)
		);

		set_transient( $cache_key, $remote, DAY_IN_SECONDS );

		if (
		is_wp_error( $remote )
		|| 200 !== wp_remote_retrieve_response_code( $remote )
		|| empty( wp_remote_retrieve_body( $remote ) )
		) {
			return false;
		}

		set_transient( $cache_key, $remote, DAY_IN_SECONDS );
	}

	$remote = json_decode( wp_remote_retrieve_body( $remote ) );

	return $remote;
}

/**
 * Function wc_waave_complete_info.
 *
 * @param string $res res.
 * @param string $action action.
 * @param array  $args args.
 */
function wc_waave_complete_info( $res, $action, $args ) {
	if ( 'plugin_information' !== $action ) {
		return $res;
	}

	if ( plugin_basename( __DIR__ ) !== $args->slug ) {
		return $res;
	}

	$remote = wc_waave_complete_request();

	if ( ! $remote ) {
		return $res;
	}

	$res = new stdClass();

	$res->name           = $remote->name;
	$res->slug           = $remote->slug;
	$res->version        = $remote->version;
	$res->tested         = $remote->tested;
	$res->requires       = $remote->requires;
	$res->author         = $remote->author;
	$res->author_profile = $remote->author_profile;
	$res->download_link  = $remote->download_url;
	$res->trunk          = $remote->download_url;
	$res->requires_php   = $remote->requires_php;
	$res->last_updated   = $remote->last_updated;

	$res->sections = array(
		'description'  => $remote->sections->description,
		'installation' => $remote->sections->installation,
		'changelog'    => $remote->sections->changelog,
	);

	if ( ! empty( $remote->banners ) ) {
		$res->banners = array(
			'low'  => $remote->banners->low,
			'high' => $remote->banners->high,
		);
	}

	return $res;
}
add_filter( 'plugins_api', 'wc_waave_complete_info', 21, 3 );

/**
 * Function waave_complete_purge.
 */
function waave_complete_purge() {
	delete_transient( 'waave_complete_update_key' );
}
add_action( 'upgrader_process_complete', 'waave_complete_purge', 10 );
